#include "MTUtilities.h"
#include "IErrors.h"
#include "MTExceptions.h"
#include "MTApplication.h"

Str255		MTFontStyle::systemFontName = { 0 };
const short	MTFontStyle::kNeedsFontNameLookup = -1;

//	MTFontStyle::MTFontStyle()
//	
//	constructor
MTFontStyle::MTFontStyle()
:id(0), size(0), face(0), mode(patOr), dirty(1)
{
	
}

//	MTFontStyle::MTFontStyle(short inID, short inSize, short inFace, short inMode)
//	
//	constructor
//	
//	inID = font id
//	inSize = size
//	inFace = face
//	inMode = drawing mode
MTFontStyle::MTFontStyle(short inID, short inSize, short inFace, short inMode)
:id(inID), size(inSize), face(inFace), mode(inMode), dirty(1)
{
	
}

//	MTFontStyle::MTFontStyle(StringPtr inName, short inSize, short inFace, short inMode)
//	
//	inName = name of font (pascal string)
//	inSize = size
//	inFace = face
//	inMode = drawing mode
MTFontStyle::MTFontStyle(StringPtr inName, short inSize, short inFace, short inMode)
:id(kNeedsFontNameLookup), size(inSize), face(inFace), mode(inMode), dirty(1)
{
	pstrcpy(name, inName);
}

//	MTFontStyle::MTFontStyle(char * inName, short inSize, short inFace, short inMode)
//	
//	inName = name of font (c-string)
//	inSize = size
//	inFace = face
//	inMode = drawing mode
MTFontStyle::MTFontStyle(char * inName, short inSize, short inFace, short inMode)
:id(kNeedsFontNameLookup), size(inSize), face(inFace), mode(inMode), dirty(1)
{
	CopyCStringToPascal(inName, name);
}

//	void MTFontStyle::CopyCurrent(CGrafPtr source)
//	
//	sets up font based on source CGrafPtr
//	
//	source = source port
void MTFontStyle::CopyCurrent(CGrafPtr source)
{
	id =	GetPortTextFont(source);
	size =	GetPortTextSize(source);
	face =	GetPortTextFace(source);
	mode =	GetPortTextMode(source);
	
	dirty = 1;
}

//	void MTFontStyle::SetCurrent(void)
//	
//	sets current font to stored font info
void MTFontStyle::SetCurrent(void)
{
	if(id == kNeedsFontNameLookup)
		LookupFontName();
	
	TextFont(id);
	TextSize(size);
	TextFace(face);
	TextMode(mode);
}

//	SInt32 MTFontStyle::GetWidth(char * buf)
//	
//	gets width of text buffer
//	
//	buf = string to measure (c-string)
SInt32 MTFontStyle::GetWidth(char * buf)
{
	MTFontSaver	saver;
	
	SetCurrent();
	
	return TextWidth(buf, 0, std::strlen(buf));
}

//	SInt32 MTFontStyle::GetWidth(StringPtr buf)
//	
//	gets width of text buffer
//	
//	buf = string to measure (pascal string)
SInt32 MTFontStyle::GetWidth(StringPtr buf)
{
	MTFontSaver	saver;
	
	SetCurrent();
	
	return StringWidth(buf);
}

//	void MTFontStyle::Update(void)
//	
//	updates font info struct (if needed)
void MTFontStyle::Update(void)
{
	if(dirty)
	{
		MTFontSaver	saver;
		
		SetCurrent();
		
		GetFontInfo(&info);
		
		dirty = 0;
	}
}

//	void MTFontStyle::LookupFontName(void)
//	
//	looks up font number from name
void MTFontStyle::LookupFontName(void)
{
	GetFNum(name, &id);
	
	if(id == 0)
	{
		if(systemFontName[0] == 0)
			LookupSystemFontName();
		
		ASSERT(EqualString(name, systemFontName, 0, 0), "MTFontStyle::LookupFontName: font not found");
	}
}

//	SInt32 Interpolate(SInt32 point, SInt32 startPoint, SInt32 endPoint, SInt32 startValue, SInt32 endValue)
//	
//	interpolates between two points
//	
//	point = point to interpolate to
//	startPoint = start point
//	endPoint = end point
//	startValue = start value
//	endValue = end value
SInt32 Interpolate(SInt32 point, SInt32 startPoint, SInt32 endPoint, SInt32 startValue, SInt32 endValue)
{
	return startValue + (((point - startPoint) * (endValue - startValue)) / (endPoint - startPoint));
}

float InterpolateFloat(float point, float startPoint, float endPoint, float startValue, float endValue)
{
	return startValue + (((point - startPoint) * (endValue - startValue)) / (endPoint - startPoint));
}

//	StringPtr pstrcpy(StringPtr to, ConstStr255Param from)
//	
//	strcpy for pascal strings
//	
//	to = destination
//	from = source
//	returns destination
StringPtr pstrcpy(StringPtr to, ConstStr255Param from)
{
	BlockMoveData(from, to, from[0] + 1);
	
	return to;
}

//	StringPtr pstrcat(StringPtr to, ConstStr255Param from)
//	
//	strcat for pascal strings
//	
//	to = destination
//	from = string to append
//	returns destination
StringPtr pstrcat(StringPtr to, ConstStr255Param from)
{
	int length = from[0];
	
	if(to[0] == 255)	// bail if we can't add any more
		return to;
	
	if((length + to[0]) > 255)	// truncate if we're going to go off the end
		length = 255 - to[0];
	
	BlockMoveData(&from[1], &to[to[0] + 1], length);	// copy the data and update the length
	to[0] += length;
	
	return to;
}

//	int pstrcmp(StringPtr a, StringPtr b)
//	
//	strcmp for pascal strings
//	
//	a = string one
//	b = string two
//	returns 0 if equal, >0 if a > b, <0 if a < b
int pstrcmp(StringPtr a, StringPtr b)
{
	if(a[0] < b[0])
		return -1;
	if(a[0] > b[0])
		return 1;
	
	return std::memcmp(&a[1], &b[1], a[0]);
}

//	UInt8 CompareFSSpec(FSSpec * a, FSSpec * b)
//	
//	compares two FSSpecs
//	
//	a = pointer to FSSpec
//	b = pointer to FSSpec
UInt8 CompareFSSpec(FSSpec * a, FSSpec * b)
{
	return	(a->vRefNum == b->vRefNum) &&
			(a->parID == b->parID) &&
			(pstrcmp(a->name, b->name) == 0);
}

UInt8 * NewSizedPtr(UInt32 * size)
{
	UInt32	bufSize = *size;
	SInt32	freeMemory = MaxBlock();
	UInt8	* buf;
	
	if(bufSize < freeMemory - 4096)
	{
		if(freeMemory <= 4096)
		{
			throw MTMemoryException("Out of memory");
		}
		else
		{
			bufSize = freeMemory - 4096;
		}
	}
	
	buf = (UInt8 *)NewPtr(bufSize);
	if(!buf)
		throw MTMemoryException("Out of memory");
	
	*size = bufSize;
	
	return buf;
}

UInt32 NavigationPutFileCustom(FSSpec * outFile, StringPtr prompt, StringPtr name, UInt8 * isReplacing)
{
	OSErr				theErr;
	NavReplyRecord		reply;
	NavDialogOptions	options;
	NavEventUPP			eventHandler;
	
	theErr = NavGetDefaultDialogOptions(&options);
	if(theErr)
		throw MTOSException(theErr, "Error getting default Navigation options");
	
	options.dialogOptionFlags &= kNavDontAddTranslateItems;
	pstrcpy(options.clientName, "\pOniTools");
	pstrcpy(options.message, prompt);
	pstrcpy(options.savedFileName, name);
	pstrcpy(options.windowTitle, "\pSave File");
	
	eventHandler = NewNavEventUPP(NavigationEventProc);
	theErr = NavPutFile(nil, &reply, &options, eventHandler, 'DATA', 'ONIt', (void *)gTheApp);
	DisposeNavEventUPP(eventHandler);
	
	if(theErr && (theErr != userCanceledErr))
		throw MTOSException(theErr, "Navigation Services error");
	
	if(reply.validRecord)
	{
		PostProcessAEDescList(&reply.selection, outFile);
		
		theErr = NavDisposeReply(&reply);
		if(theErr)
			throw MTOSException(theErr, "Navigation Services error");
		
		if(isReplacing)
			*isReplacing = reply.replacing;
		
		return 1;
	}
	
	if(isReplacing)
		*isReplacing = 0;
	
	return 0;
}

UInt32 NavigationGetFileCustom(FSSpec * outFile, StringPtr prompt)
{
	OSErr				theErr;
	NavReplyRecord		reply;
	NavDialogOptions	options;
	NavEventUPP			eventHandler;
	
	theErr = NavGetDefaultDialogOptions(&options);
	if(theErr)
		throw MTOSException(theErr, "Error getting default Navigation options");
	
	options.dialogOptionFlags &= ~kNavAllowPreviews;
	pstrcpy(options.clientName, "\pOniTools");
	pstrcpy(options.message, prompt);
	pstrcpy(options.windowTitle, "\pOpen File");
	
	eventHandler = NewNavEventUPP(NavigationEventProc);
	theErr = NavGetFile(nil, &reply, &options, eventHandler, nil, nil, nil, (void *)gTheApp);
	DisposeNavEventUPP(eventHandler);
	
	if(theErr && (theErr != userCanceledErr))
		throw MTOSException(theErr, "Navigation Services Error");
	
	if(reply.validRecord)
	{
		PostProcessAEDescList(&reply.selection, outFile);
		
		theErr = NavDisposeReply(&reply);
		if(theErr)
			throw MTOSException(theErr, "Navigation Services error");
		
		return 1;
	}
	
	return 0;
}

pascal void NavigationEventProc(NavEventCallbackMessage message, NavCBRecPtr params, NavCallBackUserData userData)
{
	MTApplication	* theApplication = (MTApplication *)userData;
	
	switch(message)
	{
		case kNavCBEvent:
		{
			switch(params->eventData.eventDataParms.event->what)
			{
				case updateEvt:
					theApplication->HandleUpdateEvent(params->eventData.eventDataParms.event);
					break;
			}
		}
		break;
	}
}

void PostProcessAEDescList(AEDescList * event, FSSpec * file)
{
	AEKeyword	keyword;
	DescType	typeCode;
	Size		theSize;
	OSErr		theErr;
	
	theErr = AEGetNthPtr(event, 1, typeFSS, &keyword, &typeCode, file, sizeof(FSSpec), &theSize);
	if(theErr)
		throw MTOSException(theErr, "Error extracting FSSpec");
}

UInt32 GetCurrentModifierState(void)
{
	EventRecord	theEvent;
	
	OSEventAvail(everyEvent, &theEvent);
	
	return theEvent.modifiers;
}
